from PyQt5.QtWidgets import (QPushButton, QWidget, QVBoxLayout, QHBoxLayout, QLabel, 
                            QComboBox, QCheckBox, QFrame, QSpinBox, QGroupBox, QLineEdit, QMessageBox)
from PyQt5.QtCore import Qt
import json
from typing import Any, Dict, List, Optional
from .ai_database import AIDatabase

# 常量定义
AI_PANEL_WIDTH = 380
AI_PANEL_Y = 50
AI_PANEL_OFFSET = 10
DEFAULT_TIMEOUT = 60
MIN_TIMEOUT = 5
MAX_TIMEOUT = 300

class AIPlugin:
    """AI插件主类，封装所有AI相关功能"""
    
    def __init__(self, main_window: Any):
        """初始化AI插件
        
        Args:
            main_window: 主窗口实例
        """
        self.main_window = main_window
        self._ai_panel = None
        self._btn_ai = None
        self._settings_page = None
        self._ai_page_index = -1
        self._original_nav_items = []
        self._original_switch_home = None
        self._original_switch_notes = None
        
        # 初始化AI独立数据库
        self._ai_db = AIDatabase()
        
        # 保存状态到主窗口
        main_window._ai_plugin = self
    
    def register(self) -> None:
        """注册AI插件功能"""
        self._add_nav_button()
        self._patch_page_methods()
        self._add_settings_tab()
    
    def _add_nav_button(self) -> None:
        """向导航栏添加AI按钮"""
        # 创建AI按钮
        btn_ai = self._create_nav_btn(" 问AI", checkable=False)
        btn_ai.clicked.connect(self._toggle_ai_panel)
        
        # 将按钮添加到导航栏布局中（右侧设置按钮之前）
        nav_layout = self.main_window._nav_widget.layout()
        settings_index = -1
        
        for i in range(nav_layout.count()):
            widget = nav_layout.itemAt(i).widget()
            if widget and hasattr(widget, "text") and widget.text().strip() == "设置":
                settings_index = i
                break
        
        if settings_index != -1:
            nav_layout.insertWidget(settings_index, btn_ai)
        else:
            nav_layout.addWidget(btn_ai)
        
        self._btn_ai = btn_ai
    
    def _create_nav_btn(self, text: str, icon_name: Optional[str] = None, checkable: bool = True) -> QPushButton:
        """创建导航按钮
        
        Args:
            text: 按钮文本
            icon_name: 图标名称（可选）
            checkable: 是否可选中
            
        Returns:
            创建的QPushButton实例
        """
        btn = QPushButton(text)
        btn.setCheckable(checkable)
        btn.setCursor(Qt.PointingHandCursor)
        if icon_name:
            # 这里可以添加图标设置逻辑
            pass
        return btn
    
    def _toggle_ai_panel(self) -> None:
        """切换AI面板的显示状态"""
        if self._ai_panel is None:
            from DPlans.ui.widgets.ai_panel import AIPanel
            self._ai_panel = AIPanel(self.main_window, self._ai_db)
        else:
            try:
                self._ai_panel._load_engines()
            except Exception as e:
                print(f"加载引擎失败: {e}")
        
        if self._ai_panel.isVisible():
            self._ai_panel.hide()
        else:
            self._place_ai_panel()
            self._ai_panel.show()
    
    def _place_ai_panel(self) -> None:
        """放置AI面板到正确位置"""
        try:
            if self._ai_panel:
                w = AI_PANEL_WIDTH
                h = self.main_window.height() - 60
                x = self.main_window.width() - w - AI_PANEL_OFFSET
                y = AI_PANEL_Y
                self._ai_panel.setGeometry(x, y, w, h)
                self._ai_panel.raise_()
        except Exception as e:
            print(f"放置AI面板失败: {e}")
    
    def _patch_page_methods(self) -> None:
        """修改页面切换方法，添加AI面板位置调整"""
        # 保存原始方法
        self._original_switch_home = self.main_window.switch_home
        self._original_switch_notes = self.main_window.switch_notes
        
        # 包装home页面切换方法
        def patched_switch_home():
            self._original_switch_home()
            self._place_ai_panel()
            try:
                if self._ai_panel:
                    self._ai_panel.raise_()
            except Exception as e:
                print(f"调整AI面板位置失败: {e}")
        
        # 包装notes页面切换方法
        def patched_switch_notes():
            self._original_switch_notes()
            self._place_ai_panel()
            try:
                if self._ai_panel:
                    self._ai_panel.raise_()
            except Exception as e:
                print(f"调整AI面板位置失败: {e}")
        
        # 替换原始方法
        self.main_window.switch_home = patched_switch_home
        self.main_window.switch_notes = patched_switch_notes
    
    def _add_settings_tab(self) -> None:
        """向设置页面添加AI选项卡"""
        from DPlans.utils.helpers import resource_path
        
        # 获取设置页面
        settings_page = self.main_window.settings_page
        self._settings_page = settings_page
        
        # 保存原始导航列表
        self._original_nav_items = [settings_page.nav_list.item(i).text() for i in range(settings_page.nav_list.count())]
        
        # 创建AI选项卡页面
        ai_page = QWidget()
        ai_v_layout = QVBoxLayout(ai_page)
        ai_v_layout.setContentsMargins(0, 0, 0, 0)
        ai_v_layout.setSpacing(20)
        
        # 标题和应用按钮行
        title_row = QHBoxLayout()
        
        # 标题
        lbl_title = QLabel("AI设置")
        lbl_title.setStyleSheet("font-size: 18px; font-weight: bold;")
        title_row.addWidget(lbl_title)
        
        # 应用和重置按钮
        btn_apply = QPushButton("应用")
        btn_apply.setStyleSheet(settings_page._btn_style_primary())
        btn_apply.clicked.connect(self._save_ai_settings)
        
        # 重置按钮
        btn_reset = QPushButton("重置")
        btn_reset.setStyleSheet("""
            QPushButton {
                border: 1px solid #f56c6c;
                background-color: #fef0f0;
                color: #f56c6c;
                border-radius: 4px;
                padding: 4px 12px;
                font-size: 14px;
            }
            QPushButton:hover { background-color: #fde2e2; }
        """)
        btn_reset.clicked.connect(self._reset_ai_settings)
        
        title_row.addStretch()
        title_row.addWidget(btn_reset)
        title_row.addWidget(btn_apply)
        
        ai_v_layout.addLayout(title_row)
        
        # 基本设置组
        basic_group = self._create_basic_settings()
        ai_v_layout.addWidget(basic_group)
        
        # 功能开关组
        func_group = self._create_function_settings()
        ai_v_layout.addWidget(func_group)
        
        # 超时设置组
        timeout_group = self._create_timeout_settings()
        ai_v_layout.addWidget(timeout_group)
        
        # 引擎管理组
        engine_group = self._create_engine_management(settings_page)
        ai_v_layout.addWidget(engine_group)
        
        # 添加到设置页面
        settings_page.stack.addWidget(ai_page)
        
        # 获取当前导航列表的长度，确定AI选项卡的索引
        self._ai_page_index = settings_page.stack.count() - 1
        
        # 向导航列表添加AI选项
        settings_page.nav_list.addItem("AI(Beta)")
    
    def _create_basic_settings(self) -> QGroupBox:
        """创建基本设置组
        
        Returns:
            创建的QGroupBox实例
        """
        basic_group = QGroupBox("基本设置")
        basic_v = QVBoxLayout(basic_group)
        
        # 预设选择
        row0 = QHBoxLayout()
        row0.addWidget(QLabel("预设:"))
        self._cb_ai_preset = QComboBox()
        self._cb_ai_preset.addItems(["自定义","DeepSeek","Kimi","Volc Ark","SiliconFlow"])
        self._cb_ai_preset.currentTextChanged.connect(self._on_ai_preset)
        row0.addWidget(self._cb_ai_preset, 1)
        basic_v.addLayout(row0)
        
        # 当前引擎选择
        row1 = QHBoxLayout()
        row1.addWidget(QLabel("当前引擎:"))
        self._cb_ai_current = QComboBox()
        row1.addWidget(self._cb_ai_current, 1)
        basic_v.addLayout(row1)
        
        return basic_group
    
    def _create_function_settings(self) -> QGroupBox:
        """创建功能开关组
        
        Returns:
            创建的QGroupBox实例
        """
        func_group = QGroupBox("功能开关")
        func_v = QVBoxLayout(func_group)
        
        # 创建AI相关的复选框
        self._cb_ai_markdown = QCheckBox("启用Markdown标记输出")
        self._cb_ai_markdown.setToolTip("启用后将格式化输出标题、表格等Markdown样式")
        func_v.addWidget(self._cb_ai_markdown)
        
        self._cb_ai_read_plans = QCheckBox("允许AI读取计划数据")
        self._cb_ai_read_plans.setToolTip("启用后AI可以读取您的计划数据，用于分析和生成建议")
        func_v.addWidget(self._cb_ai_read_plans)
        
        self._cb_ai_read_notes = QCheckBox("允许AI读取笔记数据")
        self._cb_ai_read_notes.setToolTip("启用后AI可以读取您的笔记数据，用于分析和生成建议")
        func_v.addWidget(self._cb_ai_read_notes)
        
        self._cb_ai_operations = QCheckBox("启用AI操作功能")
        self._cb_ai_operations.setToolTip("启用后AI可以创建计划、笔记和执行专注等操作")
        func_v.addWidget(self._cb_ai_operations)
        
        self._cb_ai_operation_ask = QCheckBox("开启AI操作询问")
        self._cb_ai_operation_ask.setToolTip("开启后，AI执行操作前会弹出提示窗询问用户是否执行")
        func_v.addWidget(self._cb_ai_operation_ask)
        
        self._cb_ai_timeout = QCheckBox("启用AI操作超时保护")
        self._cb_ai_timeout.setToolTip("启用后当AI操作超过设定时间会自动终止")
        func_v.addWidget(self._cb_ai_timeout)
        
        self._cb_ai_confirm = QCheckBox("执行危险操作前需要确认")
        self._cb_ai_confirm.setToolTip("如强制锁机等危险操作需要用户确认")
        func_v.addWidget(self._cb_ai_confirm)
        
        return func_group
    
    def _create_timeout_settings(self) -> QGroupBox:
        """创建超时设置组
        
        Returns:
            创建的QGroupBox实例
        """
        timeout_group = QGroupBox("超时设置")
        timeout_v = QVBoxLayout(timeout_group)
        
        timeout_row = QHBoxLayout()
        timeout_row.addWidget(QLabel("AI操作超时时间(秒):"))
        self._spin_ai_timeout = QComboBox()
        timeout_options = [30, 60, 90, 180]
        self._spin_ai_timeout.addItems([str(opt) for opt in timeout_options])
        # 默认选择60秒
        default_idx = timeout_options.index(60)
        self._spin_ai_timeout.setCurrentIndex(default_idx)
        self._spin_ai_timeout.setToolTip("AI执行操作的最大等待时间")
        timeout_row.addWidget(self._spin_ai_timeout)
        timeout_row.addStretch()
        timeout_v.addLayout(timeout_row)
        
        return timeout_group
    
    def _create_engine_management(self, settings_page: Any) -> QWidget:
        """创建引擎管理组
        
        Args:
            settings_page: 设置页面实例
            
        Returns:
            创建的QWidget实例
        """
        # 引擎数据
        self._ai_engines: List[Dict[str, str]] = []
        
        # 引擎管理容器
        container = QWidget()
        container_v = QVBoxLayout(container)
        
        # 引擎添加表单
        ai_add_frame = QFrame()
        ai_add_frame.setStyleSheet("QFrame{border:1px solid #e0e0e0;border-radius:10px;background:#ffffff;}")
        add_v = QVBoxLayout(ai_add_frame)
        add_v.setContentsMargins(12, 12, 12, 12)
        add_v.setSpacing(10)
        
        # 引擎名称
        rn = QHBoxLayout()
        rn.addWidget(QLabel("引擎名称:"))
        self._line_ai_name = QLineEdit()
        rn.addWidget(self._line_ai_name, 1)
        add_v.addLayout(rn)
        
        # Key
        rk = QHBoxLayout()
        rk.addWidget(QLabel("Key:"))
        self._line_ai_key = QLineEdit()
        self._line_ai_key.setEchoMode(QLineEdit.Password)
        rk.addWidget(self._line_ai_key, 1)
        add_v.addLayout(rk)
        
        # Base URL
        rb = QHBoxLayout()
        rb.addWidget(QLabel("Base URL:"))
        self._line_ai_base = QLineEdit()
        rb.addWidget(self._line_ai_base, 1)
        add_v.addLayout(rb)
        
        # Model
        rm = QHBoxLayout()
        rm.addWidget(QLabel("Model:"))
        self._line_ai_model = QLineEdit()
        rm.addWidget(self._line_ai_model, 1)
        add_v.addLayout(rm)
        
        # 操作按钮
        act = QHBoxLayout()
        btn_ai_add = QPushButton("添加引擎")
        btn_ai_add.setStyleSheet(settings_page._btn_style_normal())
        btn_ai_add.clicked.connect(self._add_ai_engine)
        act.addStretch()
        act.addWidget(btn_ai_add)
        add_v.addLayout(act)
        
        container_v.addWidget(ai_add_frame)
        
        # 已添加模型列表
        ai_list_frame = QFrame()
        ai_list_frame.setStyleSheet("QFrame{border:1px solid #e0e0e0;border-radius:10px;background:#ffffff;}")
        list_v = QVBoxLayout(ai_list_frame)
        list_v.setContentsMargins(12, 12, 12, 12)
        list_v.setSpacing(8)
        
        list_v.addWidget(QLabel("已添加的模型:"))
        self._ai_list_v = QVBoxLayout()
        self._ai_list_v.setContentsMargins(0, 0, 0, 0)
        self._ai_list_v.setSpacing(6)
        list_v.addLayout(self._ai_list_v)
        
        container_v.addWidget(ai_list_frame)
        
        # 加载初始设置
        self._load_ai_settings()
        
        return container
    
    def _on_ai_preset(self, name: str) -> None:
        """处理预设模型选择
        
        Args:
            name: 预设模型名称
        """
        if name == "DeepSeek":
            self._line_ai_base.setText("https://api.deepseek.com")
            self._line_ai_model.setText("deepseek-chat")
            self._line_ai_name.setText("DeepSeek")
        elif name == "Kimi":
            self._line_ai_base.setText("https://api.moonshot.cn/v1")
            self._line_ai_model.setText("kimi-latest")
            self._line_ai_name.setText("Kimi")
        elif name == "Volc Ark":
            self._line_ai_base.setText("https://ark.cn-beijing.volces.com/api/v3")
            self._line_ai_model.setText("doubao-seed-1-8-251215")
            self._line_ai_name.setText("Doubao")
        elif name == "SiliconFlow":
            self._line_ai_base.setText("https://api.siliconflow.cn/v1")
            self._line_ai_model.setText("deepseek-v3")
            self._line_ai_name.setText("SiliconFlow")
    
    def _render_engine_list(self) -> None:
        """渲染引擎列表"""
        # 清空现有列表
        while self._ai_list_v.count() > 0:
            item = self._ai_list_v.takeAt(0)
            widget = item.widget()
            if widget:
                widget.deleteLater()
        
        # 渲染新列表
        for engine in self._ai_engines:
            self._add_engine_item(engine)
    
    def _add_engine_item(self, engine: Dict[str, str]) -> None:
        """添加引擎项到列表
        
        Args:
            engine: 引擎数据
        """
        row = QHBoxLayout()
        
        name_label = QLabel(engine.get("name", ""))
        base_url_label = QLabel(engine.get("base_url", ""))
        model_label = QLabel(engine.get("model", ""))
        
        row.addWidget(name_label, 1)
        row.addWidget(base_url_label, 2)
        row.addWidget(model_label, 1)
        
        # 修改按钮
        btn_edit = QPushButton("修改")
        btn_edit.setStyleSheet(self.main_window.settings_page._btn_style_normal())
        btn_edit.clicked.connect(lambda checked, e=engine: self._edit_engine(e))
        row.addWidget(btn_edit)
        
        # 删除按钮
        btn_delete = QPushButton("删除")
        btn_delete.setStyleSheet("""
            QPushButton {
                border: 1px solid #f56c6c;
                background-color: #fef0f0;
                color: #f56c6c;
                border-radius: 4px;
                padding: 4px 12px;
                font-size: 12px;
            }
            QPushButton:hover { background-color: #fde2e2; }
        """)
        btn_delete.clicked.connect(lambda checked, n=engine.get("name", ""): self._delete_engine(n))
        row.addWidget(btn_delete)
        
        # 添加到布局
        wrap_widget = QWidget()
        wrap_widget.setLayout(row)
        self._ai_list_v.addWidget(wrap_widget)
    
    def _edit_engine(self, engine: Dict[str, str]) -> None:
        """编辑引擎
        
        Args:
            engine: 引擎数据
        """
        self._line_ai_name.setText(engine.get("name", ""))
        self._line_ai_key.setText(engine.get("key", ""))
        self._line_ai_base.setText(engine.get("base_url", ""))
        self._line_ai_model.setText(engine.get("model", ""))
    
    def _delete_engine(self, name: str) -> None:
        """删除引擎
        
        Args:
            name: 引擎名称
        """
        self._ai_engines = [e for e in self._ai_engines if e.get("name", "") != name]
        self._render_engine_list()
        self._update_current_engine_list()
    
    def _reset_ai_settings(self) -> None:
        """重置AI设置，清空所有AI数据"""
        # 确认对话框
        reply = QMessageBox.question(
            self.main_window,
            "重置AI设置",
            "确定要重置所有AI设置和数据吗？此操作不可恢复。",
            QMessageBox.Yes | QMessageBox.No,
            QMessageBox.No
        )
        
        if reply == QMessageBox.Yes:
            # 清空AI数据库
            if self._ai_db.clear_all_data():
                QMessageBox.information(
                    self.main_window,
                    "重置成功",
                    "AI设置和数据已重置为默认状态。"
                )
                # 重新加载设置
                self._load_ai_settings()
            else:
                QMessageBox.critical(
                    self.main_window,
                    "重置失败",
                    "重置AI设置和数据失败，请重试。"
                )
    
    def _add_ai_engine(self) -> None:
        """添加AI引擎"""
        name = self._line_ai_name.text().strip()
        key = self._line_ai_key.text().strip()
        base_url = self._line_ai_base.text().strip()
        model = self._line_ai_model.text().strip()
        
        if not name or not base_url or not model:
            return
        
        # 检查是否已存在同名引擎
        for engine in self._ai_engines:
            if engine.get("name", "") == name:
                # 更新现有引擎
                engine["key"] = key
                engine["base_url"] = base_url
                engine["model"] = model
                break
        else:
            # 添加新引擎
            self._ai_engines.append({
                "name": name,
                "key": key,
                "base_url": base_url,
                "model": model
            })
        
        # 重新渲染列表
        self._render_engine_list()
        
        # 更新当前引擎下拉列表
        self._update_current_engine_list()
        
        # 清空输入框
        self._line_ai_name.clear()
        self._line_ai_key.clear()
        self._line_ai_base.clear()
        self._line_ai_model.clear()
    
    def _update_current_engine_list(self) -> None:
        """更新当前引擎列表"""
        self._cb_ai_current.clear()
        self._cb_ai_current.addItem("Auto")
        for engine in self._ai_engines:
            self._cb_ai_current.addItem(engine.get("name", ""))
    
    def _load_ai_settings(self) -> None:
        """加载AI设置"""
        # 加载功能开关
        self._cb_ai_markdown.setChecked(self._ai_db.get_setting("ai_markdown", "1") == "1")
        self._cb_ai_read_plans.setChecked(self._ai_db.get_setting("ai_enable_read_plans", "1") == "1")
        self._cb_ai_read_notes.setChecked(self._ai_db.get_setting("ai_enable_read_notes", "1") == "1")
        self._cb_ai_operations.setChecked(self._ai_db.get_setting("ai_operations", "1") == "1")
        self._cb_ai_operation_ask.setChecked(self._ai_db.get_setting("ai_operation_ask", "1") == "1")
        self._cb_ai_timeout.setChecked(self._ai_db.get_setting("ai_timeout", "1") == "1")
        self._cb_ai_confirm.setChecked(self._ai_db.get_setting("ai_confirm", "1") == "1")
        
        # 加载超时设置
        try:
            timeout_value = int(self._ai_db.get_setting("ai_timeout_seconds", str(DEFAULT_TIMEOUT)))
            # 查找对应的索引
            idx = self._spin_ai_timeout.findText(str(timeout_value))
            if idx != -1:
                self._spin_ai_timeout.setCurrentIndex(idx)
            else:
                # 如果找不到，默认选择60秒
                self._spin_ai_timeout.setCurrentIndex(self._spin_ai_timeout.findText("60"))
        except ValueError:
            # 默认选择60秒
            self._spin_ai_timeout.setCurrentIndex(self._spin_ai_timeout.findText("60"))
        
        # 加载引擎设置
        raw_engines = self._ai_db.get_setting("ai_engines", "")
        if raw_engines:
            try:
                self._ai_engines = json.loads(raw_engines)
            except json.JSONDecodeError as e:
                print(f"解析引擎设置失败: {e}")
                self._ai_engines = []
        else:
            self._ai_engines = []
        
        # 渲染引擎列表
        self._render_engine_list()
        
        # 更新当前引擎下拉列表
        self._update_current_engine_list()
        
        # 设置当前引擎
        current_engine = self._ai_db.get_setting("ai_current_engine", "Auto")
        idx = self._cb_ai_current.findText(current_engine) if current_engine else 0
        self._cb_ai_current.setCurrentIndex(max(0, idx))
    
    def _save_ai_settings(self) -> None:
        """保存AI设置"""
        # 保存功能开关
        self._ai_db.set_setting("ai_markdown", "1" if self._cb_ai_markdown.isChecked() else "0")
        self._ai_db.set_setting("ai_enable_read_plans", "1" if self._cb_ai_read_plans.isChecked() else "0")
        self._ai_db.set_setting("ai_enable_read_notes", "1" if self._cb_ai_read_notes.isChecked() else "0")
        self._ai_db.set_setting("ai_operations", "1" if self._cb_ai_operations.isChecked() else "0")
        self._ai_db.set_setting("ai_operation_ask", "1" if self._cb_ai_operation_ask.isChecked() else "0")
        self._ai_db.set_setting("ai_timeout", "1" if self._cb_ai_timeout.isChecked() else "0")
        self._ai_db.set_setting("ai_confirm", "1" if self._cb_ai_confirm.isChecked() else "0")
        
        # 保存超时设置
        self._ai_db.set_setting("ai_timeout_seconds", self._spin_ai_timeout.currentText())
        
        # 保存引擎设置
        self._ai_db.set_setting("ai_engines", json.dumps(self._ai_engines))
        self._ai_db.set_setting("ai_current_engine", self._cb_ai_current.currentText())
    
    def _patch_page_methods(self) -> None:
        """修改页面切换方法"""
        # 保存原始方法
        self._original_switch_home = self.main_window.switch_home
        self._original_switch_notes = self.main_window.switch_notes
        
        # 包装home页面切换方法
        def patched_switch_home():
            self._original_switch_home()
            self._place_ai_panel()
            try:
                if self._ai_panel:
                    self._ai_panel.raise_()
            except Exception as e:
                print(f"调整AI面板位置失败: {e}")
        
        # 包装notes页面切换方法
        def patched_switch_notes():
            self._original_switch_notes()
            self._place_ai_panel()
            try:
                if self._ai_panel:
                    self._ai_panel.raise_()
            except Exception as e:
                print(f"调整AI面板位置失败: {e}")
        
        # 替换原始方法
        self.main_window.switch_home = patched_switch_home
        self.main_window.switch_notes = patched_switch_notes
    
    def unregister(self) -> None:
        """卸载AI插件功能"""
        self._remove_nav_button()
        self._restore_page_methods()
        self._remove_settings_tab()
        self._cleanup_ai_panel()
    
    def _remove_nav_button(self) -> None:
        """移除导航栏中的AI按钮"""
        if self._btn_ai:
            nav_layout = self.main_window._nav_widget.layout()
            nav_layout.removeWidget(self._btn_ai)
            self._btn_ai.deleteLater()
            self._btn_ai = None
    
    def _restore_page_methods(self) -> None:
        """恢复原始页面切换方法"""
        if self._original_switch_home:
            self.main_window.switch_home = self._original_switch_home
            self._original_switch_home = None
        
        if self._original_switch_notes:
            self.main_window.switch_notes = self._original_switch_notes
            self._original_switch_notes = None
    
    def _remove_settings_tab(self) -> None:
        """从设置页面移除AI选项卡"""
        settings_page = self._settings_page
        if settings_page:
            # 移除AI选项卡页面
            ai_page = settings_page.stack.widget(self._ai_page_index)
            if ai_page:
                settings_page.stack.removeWidget(ai_page)
                ai_page.deleteLater()
            
            # 移除导航列表中的AI选项
            # 先清空导航列表
            settings_page.nav_list.clear()
            # 再重新添加原始的导航项
            for item in self._original_nav_items:
                settings_page.nav_list.addItem(item)
            
            # 确保当前选中的索引有效
            current_index = settings_page.nav_list.currentRow()
            if current_index >= settings_page.nav_list.count():
                settings_page.nav_list.setCurrentRow(0)
    
    def _cleanup_ai_panel(self) -> None:
        """清理AI面板"""
        if self._ai_panel:
            self._ai_panel.hide()
            self._ai_panel.deleteLater()
            self._ai_panel = None

# 插件注册时调用的函数
def register(main_window: Any) -> None:
    """
    注册AI插件功能到主程序
    
    Args:
        main_window: 主窗口实例
    """
    plugin = AIPlugin(main_window)
    plugin.register()

# 插件卸载时调用的函数
def unregister(main_window: Any) -> None:
    """
    从主程序中移除AI插件功能
    
    Args:
        main_window: 主窗口实例
    """
    if hasattr(main_window, "_ai_plugin"):
        plugin = main_window._ai_plugin
        plugin.unregister()
        del main_window._ai_plugin
